/* -*- mode: C++; tab-width: 4 -*- */
/* ===================================================================== *\
	Copyright (c) 2000-2001 Palm, Inc. or its subsidiaries.
	All rights reserved.

	This file is part of the Palm OS Emulator.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
\* ===================================================================== */

#include "EmCommon.h"
#include "EmRegsS1D13A03.h"

#include "EmScreen.h"			// EmScreen::InvalidateAll
#include "EmPixMap.h"			// EmPixMap::GetLCDScanlines
#include "SessionFile.h"		// WriteS1D13A03RegsType
#include "EmMemory.h"			// EmMem_memcpy


// Given a register (specified by its field name), return its address
// in emulated space.

#define addressof(reg)				\
	(this->GetAddressStart () + fRegs.offsetof_##reg ())


// Macro to help the installation of handlers for a register.

#define INSTALL_HANDLER(read, write, reg)			\
	this->SetHandler (	(ReadFunction) &EmRegsS1D13A03::read,		\
						(WriteFunction) &EmRegsS1D13A03::write,		\
						addressof (reg),			\
						fRegs.reg.GetSize ())

// Panel type register [0ch]
//#define s1d13a03MonoMask			0x40

// Display mode register [10h]
//#define s1d13a03DisplayBlankMask  0x400000
//#define s1d13a03BPPMask			0x1F
//#define s1d13a03BPPShift		    0x00

// Special effects register [60h]
//#define s1d13a032DSwapMask			0x80
//#define s1d13a03WordSwapMask			0x40
#define s1d13a03ByteSwapMask			0x00000020
//#define s1d13a03SubWindowEnableMask	0x10

// [64h]
//#define hwrDisplayGPIOPCI			0x80			// in GPIOStatusControl0
//#define hwrDisplayGPIOEL_ON		0x40			// in GPIOStatusControl0
#define hwrDisplayGPIOLCD_ON		0x00000020		// in GPIOStatusControl0
//#define hwrDisplayGPIOMOD			0x10			// in GPIOStatusControl0

// REG[0x00] Product/Revision Code Register
// bits 7-2 are product code, bits 1-0 are revision code
#define s1d13a03ProductCodeMask					0xFC
#define s1d13a03RevisionCodeMask				0x03
#define s1d13a03ProductCode						0x0B
#define s1d13a03ProductCodeExpected				(s1d13a03ProductCode << 2)
#define s1d13a03RevisionCodeExpected			0x00


// REG[0x05] pixelClkConfig register
#define	s1d13a03PixelClockDivide8				0x40
#define	s1d13a03PixelClockDivide4				0x30
#define	s1d13a03PixelClockDivide3				0x20
#define	s1d13a03PixelClockDivide2				0x10
#define	s1d13a03PixelClockDivide1				0x00

#define	s1d13a03PixelClockClkI2					0x03
#define	s1d13a03PixelClockClkI					0x02
#define	s1d13a03PixelClockBClkI					0x01
#define	s1d13a03PixelClockMClkI					0x00

// REG[0x70] displayMode register
#define	s1d13a03DisplayModeBlank				0x00800000
#define	s1d13a03DisplayModeDitherOff			0x40
#define	s1d13a03DisplayModeHwInvert				0x20
#define	s1d13a03DisplayModeSwInvert				0x10
#define	s1d13a03DisplayModeDepthMask			0x07
#define	s1d13a03DisplayModeDepth16				0x10
#define	s1d13a03DisplayModeDepth8				0x08
#define	s1d13a03DisplayModeDepth4				0x04
#define	s1d13a03DisplayModeDepth2				0x02
#define	s1d13a03DisplayModeDepth1				0x01


// SED13A3 REG[0x14] Power Save Configuration Register
#define	s1d13a03PowerSaveVNonDisplayStatus		0x80
#define	s1d13a03PowerSaveEnable					0x10

#define s1d13a03GpioStatusControl1PanelEnable	0x01

// REG[0xB0] PWM/Contrast Voltage Clock Control
#define	s1d13a03PwmCvClockControlEnablePWM		0x10

// REG[0xB1] PWM/Contrast Voltage configuration
#define	s1d13a03PwmCvClockConfigDiv4096			0xC0
#define	s1d13a03PwmCvClockConfigDiv2048			0xB0
#define	s1d13a03PwmCvClockConfigDiv1024			0xA0
#define	s1d13a03PwmCvClockConfigDiv512			0x90
#define	s1d13a03PwmCvClockConfigDiv256			0x80
#define	s1d13a03PwmCvClockConfigDiv128			0x70
#define	s1d13a03PwmCvClockConfigDiv64			0x60
#define	s1d13a03PwmCvClockConfigDiv32			0x50
#define	s1d13a03PwmCvClockConfigDiv16			0x40
#define	s1d13a03PwmCvClockConfigDiv8			0x30
#define	s1d13a03PwmCvClockConfigDiv4			0x20
#define	s1d13a03PwmCvClockConfigDiv2			0x10
#define	s1d13a03PwmCvClockConfigDiv1			0x00

#define	s1d13a03PwmCvClockConfigClkI			0x00
#define	s1d13a03PwmCvClockConfigClkI2			0x01


// REG[0xAC]  GPIO Status/Control
#define	s1d13a03GpioStatusControl0PanelPwr1Off	0x10	// low to supply power
#define	s1d13a03GpioStatusControl0PanelPwr2On	0x20	// high to supply power

// REG[0xAD]  GPO Status/Control
//#define	s1d13a03GpioStatusControl1PanelEnable	0x80	// high to enable panel

// (vjc) 13A3-specific masks:
// REG 0x0004,	Memory Clock Configuration Register
#define s1d13a03MemClockConfigMClkDiv1			0x00
#define s1d13a03MemClockConfigMClkDiv2			0x01
#define s1d13a03MemClockConfigMClkDiv3			0x02
#define s1d13a03MemClockConfigMClkDiv4			0x03

// REG 0x0008,	Pixel Clock Configuration Register
#define s1d13a03PixelClockConfigMClk				0x00
#define s1d13a03PixelClockConfigBClk				0x01
#define s1d13a03PixelClockConfigClkI				0x02
#define s1d13a03PixelClockConfigClkI2			    0x03

#define s1d13a03PixelClockConfigDiv1				0x00
#define s1d13a03PixelClockConfigDiv2				0x10
#define s1d13a03PixelClockConfigDiv3				0x20
#define s1d13a03PixelClockConfigDiv4				0x30
#define s1d13a03PixelClockConfigDiv8				0x40


// REG 0x0073, PWM Configuration Register
#define s1d13a03PWMConfigClockDivideSelectMask	    0xF0
#define s1d13a03PWMConfigClockDivideSelectOffset	4
#define s1d13a03PWMConfigClockForceHi			    0x08
#define s1d13a03PWMConfigClockSourceSelectMask	    0x06
#define s1d13a03PWMConfigClockSourceSelectOffset	1
#define s1d13a03PWMConfigClockEnable				0x01

#define s1d13a03PWMConfigClockSourceClkI			0x00
#define s1d13a03PWMConfigClockSourceClkI2		    0x01
#define s1d13a03PWMConfigClockSourceBClk			0x02
#define s1d13a03PWMConfigClockSourcePClk			0x03

// REG 0x0077, PWM Duty Cycle Register


// REG[0x10] displaySettings register
#define	sed13A03DisplaySettingsDitheringDisable		0x0040

#define s1d13a03DsplySettingsPiP					0x0008

#define s1d13a03DsplySettingsSwivelNone				0x0000
#define s1d13a03DsplySettingsSwivel90				0x0001
#define s1d13a03DsplySettingsSwivel180				0x0002
#define s1d13a03DsplySettingsSwivel270				0x0003

#define s1d13a03DsplySettingsInverseVideo			0x0010

#define s1d13a03FPLinePolarity						0x800000

// FPLINE Pulse Width can be 0 to 63, but 7 seems to work best:
#define s1d13a03FPlineHorizSyncPulseWidthOffset	16
#define s1d13a03FPlineHorizSyncPulseWidth(WIDTH)	((WIDTH) << s1d13a03FPlineHorizSyncPulseWidthOffset)

#define s1d13a03FPlinePulseStartPosition			0

#define s1d13a03FPFramePolarity					0x800000

// FPFRAME Pulse Width can be 0 to 7:
#define s1d13a03FPFramePulseWidthOffset			16
#define s1d13a03FPFramePulseWidth(WIDTH)			(L((WIDTH) << s1d13a03FPlineHorizSyncPulseWidthOffset))

#define s1d13a03FPFramePulseStartPosition		1

#define s1d13a03PiPXEndPosition					160
#define s1d13a03PiPXStartPosition				1

#define s1d13a03PiPYEndPosition					160
#define s1d13a03PiPYStartPosition				1


// The current S1D13A03 driver code is completely whacked.  It's initialized to use
// the main display for drawing (see HwrDisplayInit in HwrDisplayBootS1D13A03.c), but
// if you change the base address, it (a) always returns the base of the embedded
// SRAM buffer as the old base address, and (b) stores the new base address in
// ovlyStartAddress (see PrvDisplayBaseAddr in HwrDisplayS1D13A03.c).  Also, the
// splash screen appears to be drawn relative to ovlyStartAddress.  Finally,
// there appears to be code to draw a border in the main screen around the
// overlay screen (see PrvDisplayBorder in HwrDisplayBootS1D13A03.c), however,
// the mainStartAddress registers are never changed to point to this border.
// On top of that, the border is only programmed on the left and right of the
// of the overlay screen, not on the top and bottom.

#define OVERLAY_IS_MAIN			1

// Given a register (specified by its field name), return its address
// in emulated space.

#define addressof(reg)				\
	(this->GetAddressStart () + fRegs.offsetof_##reg ())

// Macro to help the installation of handlers for a register.

#define INSTALL_HANDLER(read, write, reg)			\
	this->SetHandler (	(ReadFunction) &EmRegsS1D13A03::read,		\
						(WriteFunction) &EmRegsS1D13A03::write,		\
						addressof (reg),			\
						fRegs.reg.GetSize ())

// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::EmRegsS1D13A03
// ---------------------------------------------------------------------------

EmRegsS1D13A03::EmRegsS1D13A03 (emuptr baseRegsAddr, emuptr baseVideoAddr) :
	fBaseRegsAddr (baseRegsAddr),
	fBaseVideoAddr (baseVideoAddr),
	fRegs ()
{
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::~EmRegsS1D13A03
// ---------------------------------------------------------------------------

EmRegsS1D13A03::~EmRegsS1D13A03 (void)
{
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::Initialize
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::Initialize (void)
{
	EmRegs::Initialize ();
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::Reset
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::Reset (Bool hardwareReset)
{
	EmRegs::Reset (hardwareReset);

	if (hardwareReset)
	{
		memset (fRegs.GetPtr (), 0, fRegs.GetSize ());

//		assert ((s1d13a03ProductCodeExpected | s1d13a03RevisionCodeExpected) == 0x0B);
		fRegs.productInformation = 0x0B1E1C0B;
		//fRegs.displayBufferSize		= 20;	// 80K / 4K
		//fRegs.configurationReadback	= 0;
	}
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::Save
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::Save (SessionFile& f)
{
	EmRegs::Save (f);

	f.WriteS1D13A03RegsType (*(S1D13A03RegsType*) fRegs.GetPtr ());
	f.WriteS1D13A03Palette (fClutData);
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::Load
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::Load (SessionFile& f)
{
	EmRegs::Load (f);

	// Read in the SED registers.

	if (!f.ReadS1D13A03RegsType (*(S1D13A03RegsType*) fRegs.GetPtr ()))
	{
		f.SetCanReload (false);
	}

	// Read in the LCD palette.

	if (!f.ReadS1D13A03Palette (fClutData))
	{
		f.SetCanReload (false);
	}
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::Dispose
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::Dispose (void)
{
	EmRegs::Dispose ();
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::SetSubBankHandlers
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::SetSubBankHandlers (void)
{
	// Install base handlers.

	EmRegs::SetSubBankHandlers ();

	// Now add standard/specialized handers for the defined registers.

	INSTALL_HANDLER (StdReadBE,			NullWrite,				productInformation);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				memoryClockConfiguration);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				pixelClockConfiguration);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				panelTypeAndMODRate);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		displaySettings);
	INSTALL_HANDLER (powerSaveConfigurationRead,			StdWriteBE,				powerSaveConfiguration);
	INSTALL_HANDLER (StdReadBE,			lutWriteAddressWrite,	lutWriteAddress);
	INSTALL_HANDLER (StdReadBE,			lutReadAddressWrite,	lutReadAddress);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				horizontalTotal);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		horizontalPeriod);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				horizontalPeriodStart);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				FPLINE);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				verticalTotal);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		verticalPeriod);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				verticalPeriodStart);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				FPFRAME);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		mainStartAddress);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		mainLineAddressOffset);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		ovlyStartAddress);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		ovlyLineAddressOffset);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		ovlyXPositions);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		ovlyYPositions);
	INSTALL_HANDLER (StdReadBE,			invalidateWriteX,		specialPurpose);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				GPIOStatusControl);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				PWMClockConfig);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				PWMOutDutyCycle);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				scratchPadA);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				scratchPadB);
	INSTALL_HANDLER (StdReadBE,			StdWriteBE,				scratchPadC);
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::GetRealAddress
// ---------------------------------------------------------------------------

uint8* EmRegsS1D13A03::GetRealAddress (emuptr address)
{
	return (uint8*) fRegs.GetPtr () + address - this->GetAddressStart ();
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::GetAddressStart
// ---------------------------------------------------------------------------

emuptr EmRegsS1D13A03::GetAddressStart (void)
{
	return fBaseRegsAddr;
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::GetAddressRange
// ---------------------------------------------------------------------------

uint32 EmRegsS1D13A03::GetAddressRange (void)
{
	return fRegs.GetSize ();
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::GetLCDScreenOn
// ---------------------------------------------------------------------------

Bool EmRegsS1D13A03::GetLCDScreenOn (void)
{
//	return ((fRegs.displaySettings) & s1d13a03DisplayModeBlank) == 0;
//	return ((fRegs.GPIOStatusControl) & hwrDisplayGPIOLCD_ON) != 0;
	return true;
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::GetLCDBacklightOn
// ---------------------------------------------------------------------------

Bool EmRegsS1D13A03::GetLCDBacklightOn (void)
{
	return true;
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::GetLCDHasFrame
// ---------------------------------------------------------------------------

Bool EmRegsS1D13A03::GetLCDHasFrame (void)
{
	return true;
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::invalidateWriteX
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::invalidateWriteX (emuptr address, int size, uint32 value)
{
	this->StdWriteBE (address, size, value);
	EmScreen::InvalidateAll ();
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::powerSaveConfigurationRead
// ---------------------------------------------------------------------------

uint32 EmRegsS1D13A03::powerSaveConfigurationRead (emuptr address, int size)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

	// Always set the vertical non-display status high since in the real
	// hardware, the ROM will check this flag in order to write the CLUT
	// registers.

	return (fRegs.powerSaveConfiguration) | 0x00000080;
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::lutWriteAddressWrite
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::lutWriteAddressWrite (emuptr address, int size, uint32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
    uint32 new_val = 0;

	new_val = value & 0xFF000000;
	new_val = new_val >> 24;

	uint8	red	    = (value >> 16) & 0x00FF;
	uint8	green	= (value >> 8) & 0x0000FF;
	uint8	blue	= (value & 0x000000FF);

/*	fClutData[new_val] = RGBType ((red & 0xFC) >> 2,
								(green & 0xFC) >> 2,
								(blue & 0xFC) >> 2);
*/
	fClutData[new_val] = RGBType ((red & 0xFC),
								(green & 0xFC),
								(blue & 0xFC));


	EmScreen::InvalidateAll ();
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::lutReadAddressWrite
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::lutReadAddressWrite (emuptr address, int size, uint32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)
	uint32 lookupIdx = 0;

	lookupIdx = value & 0x000000FF;
//	new_val = new_val >> 24;

	RGBType	rgb = fClutData[lookupIdx];

	fRegs.lutReadAddress = lookupIdx << 24;
//	fRegs.lutReadAddress = fRegs.lutReadAddress | (((rgb.fRed & 0xFC) << 16) | ((rgb.fGreen & 0xFC) << 8) | (rgb.fBlue & 0xFC));
	uint32 red = rgb.fRed & 0xFC;
	uint32 green = rgb.fGreen & 0xFC;
	uint32 blue = rgb.fBlue & 0xFC;
	fRegs.lutReadAddress = fRegs.lutReadAddress | (red << 16) | (green << 8) | blue;

}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03::PrvGetPalette
// ---------------------------------------------------------------------------

void EmRegsS1D13A03::PrvGetPalette (RGBList& thePalette)
{
	Bool	mono		= ((fRegs.displaySettings << 16) & 0x0040) != 0;
	int32	bpp			= fRegs.displaySettings & 0x0000001F;
//	int32	bpp			= 1 << ((fRegs.displaySettings & 0x0000001F) >> 0x00);
	bpp = s1d13a03DisplayModeDepth8;
	int32	numColors	= 1 << bpp;

	thePalette.resize (numColors);

	for (int ii = 0; ii < numColors; ++ii)
	{
		if (mono)
		{
			uint8	green = fClutData[ii].fGreen;
			thePalette[ii].fRed		= green;
			thePalette[ii].fGreen	= green;
			thePalette[ii].fBlue	= green;
		}
		else
		{
			thePalette[ii] = fClutData[ii];
		}
	}
}


#pragma mark -

// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::EmRegsS1D13A03Morf
// ---------------------------------------------------------------------------

EmRegsS1D13A03Morf::EmRegsS1D13A03Morf (emuptr baseRegsAddr,
												  emuptr baseVideoAddr) :
	EmRegsS1D13A03 (baseRegsAddr, baseVideoAddr)
{
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::~EmRegsS1D13A03Morf
// ---------------------------------------------------------------------------

EmRegsS1D13A03Morf::~EmRegsS1D13A03Morf (void)
{
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::SetSubBankHandlers
// ---------------------------------------------------------------------------

void EmRegsS1D13A03Morf::SetSubBankHandlers (void)
{
	// Install base handlers.

	EmRegsS1D13A03::SetSubBankHandlers ();

	// Now add standard/specialized handers for the defined registers.

#undef INSTALL_HANDLER
#define INSTALL_HANDLER(read, write, reg)			\
	this->SetHandler (	(ReadFunction) &EmRegsS1D13A03Morf::read,		\
						(WriteFunction) &EmRegsS1D13A03Morf::write,		\
						addressof (reg),			\
						fRegs.reg.GetSize ())

//	INSTALL_HANDLER (StdReadBE,			reservedWrite,			reserved);
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::GetLCDBeginEnd
// ---------------------------------------------------------------------------

void EmRegsS1D13A03Morf::GetLCDBeginEnd (emuptr& begin, emuptr& end)
{
	// Get the screen metrics.

	// The hardware is written to in reverse, so the mainStartOffsetX registers
	// report the END of the frame buffer, not the beginning.
	emuptr	baseAddr	= fBaseVideoAddr;

	int32	width		= (((fRegs.horizontalPeriod & 0x0000007F) + 1) * 8);
	int32	height		= (fRegs.verticalPeriod & 0x000003FF) + 1;
	width = 160;
	height = 160;
	int32	rowBytes	= ((width * this->PrvGetLCDDepth ()) / 8);

	begin = baseAddr;
	end = baseAddr + rowBytes * height;
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::GetLCDScanlines
// ---------------------------------------------------------------------------

void EmRegsS1D13A03Morf::GetLCDScanlines (EmScreenUpdateInfo& info)
{
	// Get the screen metrics.

	Bool	byteSwapped	= (fRegs.specialPurpose & s1d13a03ByteSwapMask) != 0;
	Bool	mono		= ((fRegs.displaySettings << 16) & 0x0040) != 0;;
//	int32	bpp			= 1 << ((fRegs.displaySettings & 0x0000001F) >> 0x00);
	int32	bpp			= (fRegs.displaySettings & 0x0000001F);

	// The hardware is written to in reverse, so the mainStartOffsetX registers
	// report the END of the frame buffer, not the beginning.
	emuptr	baseAddr	= fBaseVideoAddr;

	int32	width		= (((fRegs.horizontalPeriod & 0x0000007F) + 1) * 8);
	int32	height		= (fRegs.verticalPeriod & 0x000003FF) + 1;
	width = 160;
	height = 160;
	int32	rowBytes	= ((width * this->PrvGetLCDDepth ()) / 8);

	info.fLeftMargin	= 0;

	if (bpp <= 8)
	{
		EmPixMapFormat	format	=	bpp == 1 ? kPixMapFormat1 :
									bpp == 2 ? kPixMapFormat2 :
									bpp == 4 ? kPixMapFormat4 :
									kPixMapFormat8;

		RGBList	colorTable;
		this->PrvGetPalette (colorTable);

		// Set format, size, and color table of EmPixMap.

		info.fImage.SetSize			(EmPoint (width, height));
		info.fImage.SetFormat		(format);
		info.fImage.SetRowBytes		(rowBytes);
		info.fImage.SetColorTable	(colorTable);

		// Determine first and last scanlines to fetch, and fetch them.

		info.fFirstLine		= (info.fScreenLow - baseAddr) / rowBytes;
		info.fLastLine		= (info.fScreenHigh - baseAddr - 1) / rowBytes + 1;

		long	firstLineOffset	= info.fFirstLine * rowBytes;
		long	lastLineOffset	= info.fLastLine * rowBytes;

		EmMem_memcpy (
			(void*) ((uint8*) info.fImage.GetBits () + firstLineOffset),
			baseAddr + firstLineOffset,
			lastLineOffset - firstLineOffset);
	}
	else
	{
		// Set depth, size, and color table of EmPixMap.

		info.fImage.SetSize (EmPoint (width, height));
		info.fImage.SetFormat (kPixMapFormat24RGB);

		// Determine first and last scanlines to fetch.

		info.fFirstLine		= (info.fScreenLow - baseAddr) / rowBytes;
		info.fLastLine		= (info.fScreenHigh - baseAddr - 1) / rowBytes + 1;

		// Get location and rowBytes of source bytes.

		uint8*	srcStart		= EmMemGetRealAddress (baseAddr);
		int32	srcRowBytes		= rowBytes;
		uint8*	srcPtr			= srcStart + srcRowBytes * info.fFirstLine;
		uint8*	srcPtr0			= srcPtr;

		// Get location and rowBytes of destination bytes.

		uint8*	destStart		= (uint8*) info.fImage.GetBits ();
		int32	destRowBytes	= info.fImage.GetRowBytes ();
		uint8*	destPtr			= destStart + destRowBytes * info.fFirstLine;
		uint8*	destPtr0		= destPtr;

		// Get height of range to copy.

		int32	height = info.fLastLine - info.fFirstLine;

		// Copy the pixels from source to dest.

		for (int yy = 0; yy < height; ++yy)
		{
			for (int xx = 0; xx < width; ++xx)
			{
				uint8	p1 = EmMemDoGet8 (srcPtr++);	// GGGBBBBB
				uint8	p2 = EmMemDoGet8 (srcPtr++);	// RRRRRGGG

				// Merge the two together so that we get RRRRRGGG GGGBBBBB

				uint16	p;

				if (!byteSwapped)
					p = (p2 << 8) | p1;
				else
					p = (p1 << 8) | p2;

				// Shift the bits around, forming RRRRRrrr, GGGGGGgg, and
				// BBBBBbbb values, where the lower-case bits are copies of
				// the least significant bits in the upper-case bits.
				//
				// Note that all of this could also be done with three 64K
				// lookup tables.  If speed is an issue, we might want to
				// investigate that.

				if (mono)
				{
					uint8	green = ((p >> 3) & 0xFC) | ((p >>  5) & 0x03);
					*destPtr++ = green;
					*destPtr++ = green;
					*destPtr++ = green;
				}
				else
				{
					*destPtr++ = ((p >> 8) & 0xF8) | ((p >> 11) & 0x07);
					*destPtr++ = ((p >> 3) & 0xFC) | ((p >>  5) & 0x03);
					*destPtr++ = ((p << 3) & 0xF8) | ((p >>  0) & 0x07);
				}
			}

			srcPtr	= srcPtr0 += srcRowBytes;
			destPtr	= destPtr0 += destRowBytes;
		}
	}
}


// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::PrvGetLCDDepth
// ---------------------------------------------------------------------------

int32 EmRegsS1D13A03Morf::PrvGetLCDDepth (void)
{
	UInt8	depth = fRegs.displaySettings & 0x0000001F;

	//depth &= s1d13a03DisplayModeDepthMask;

	switch (depth)
	{
		case s1d13a03DisplayModeDepth16: return 16;
		case s1d13a03DisplayModeDepth8:  return 8;
		case s1d13a03DisplayModeDepth4:  return 4;
		case s1d13a03DisplayModeDepth2:  return 2;
		case s1d13a03DisplayModeDepth1:  return 1;
		//default: EmAssert (false);
	}

	return 8;
}

 
// ---------------------------------------------------------------------------
//		 EmRegsS1D13A03Morf::reservedWrite
// ---------------------------------------------------------------------------

void EmRegsS1D13A03Morf::reservedWrite (emuptr address, int size, uint32 value)
{
	UNUSED_PARAM (address);
	UNUSED_PARAM (size);
	UNUSED_PARAM (value);
}
